*-------------------------------------------------------------------------------
/* title : Estimation of production function under the assumption of Impefect
		   Competition (IC) in the output market, i.e. monopolistic competition
		   and CES preferences.
   input : $temp/Sample.dta
   output: all relevant point estimates */
*-------------------------------------------------------------------------------

*===============================================================================
*----------IMPERFECT COMPETITION (MONOP. COMPETITION & CES PREFERENCES)---------
*===============================================================================
*Aggregate demand shifter
gegen Yt = sum(Y*(SALES/sum(SALES))), by(country $indlev year) 	// Market-share weighted average of deflated sales.
//gegen Yt = mean(Y), by(country $indlev year) 					// Simple average of deflated sales
gen yt = ln(Yt)

*===============================================================================
*----------------------FIRST STEP (SHARE EQUATION-NLLS)-------------------------
*===============================================================================
glevelsof $indlev, local(industries) 			// Declare local with industry list
xtset newid year 								// Identify panel structure

*------------------------OLS STARTING VALUES FOR NLLS---------------------------
foreach i of local industries {
	gen ${indlev}_`: label ($indlev) `i'' = $indlev == `i'
}
qui reg share ${indlev}_*, nocons 				// OLS regression

*-------------------------------NLLS REGRESSORS---------------------------------
foreach i of local industries {
	local nl_var `nl_var'`plus'{g_o_`i' = -_b[${indlev}_`i']}*${indlev}_`i' // Regressors for the Non-linear Least Square estimation below
	local plus +
}

*-------------TIME DUMMIES IN NLLS CAPTURING TIME-VARYING "MARKUPS--------------
qui tab year, gen(d_)
replace d_1 = 0 								// To avoid dummy variable trap exclude 1st period dummy that "enters" in the ln() as delta!
foreach var of varlist d_* {
	local dummies `dummies'+{`var'}*`var'
}

*-------------------------NON-LINEAR LEAST SQARE ESTIMATION---------------------
nl (share = ln(`nl_var') `dummies'), noconstant // Non-linear Least Square estimation to minimise the sum of squared residuals

*--------------------------------SIGNAL EXTRACTION------------------------------
predict lnDhat_Ehat_delta_dt					// ln of output elasticity of materials up to a constant lnE, delta and dt
gen eit 	= - share + lnDhat_Ehat_delta_dt
gen exp_eit = exp(eit)
sum exp_eit, mean
scalar Ehat = r(mean)							// Sample analogue of E[exp^(eit)]

*-----------------------TIME-VARYING MARKUPS UP TO delta------------------------
qui sum year
local N = r(max) - r(min) + 1
local plus 										// Ensure local is empty
forvalues t = 1/`N' {
	local dt `dt'`plus'_b[/d_`t']*d_`t' 
	local plus +
}
gen dt = `dt' 									// Rowtotal of the per period dt: growth pattern of markups (not levels) up to a constant delta

*----OUTPUT ELASTICITY OF MATERIALS FREE OF E=E[exp^(eit)], dt, & UP TO delta---
gen a_m_delta = exp(lnDhat_Ehat_delta_dt-ln(Ehat)-dt) //Output elasticity of materials free of constant Epsilon=E[exp^(eit)] and dt, and up to constant delta

*----------COEFFICIENTS UP TO delta & FREE OF CONSTANT E=E[exp^(eit)]-----------
foreach var in o {
	local plus 									// Ensure local is empty
	foreach i of local industries {
		local g_`var' `g_`var''`plus'(_b[/g_`var'_`i']/Ehat)*${indlev}_`i'
		local plus +
	}
	gen g_`var'_delta = `g_`var'' 				// Coefficients up to delta & free of constant E=E[exp^(eit)]. For parametric pf, they coincide with the respective pf parameters up to delta
}

*--------------------SAMPLE ANALOGUE OF OBSERVED PART OF PF---------------------
gen Rhat = y - eit - exp(dt)*m*(g_o_delta)

*-------------------------LAGS OF RELEVANT VARIABLES----------------------------
qui foreach var of varlist Rhat eit yt { 		// For second-step GMM
	gen L1_`var'	= L.`var'
	gen L2_`var'	= L2.`var'
	gen D_`var' 	= D.`var'
	gen DL1_`var' 	= D.L1_`var'
}

//sum a_m_delta SHARE exp_eit dt
drop Yt d_* lnDhat_Ehat* exp_eit g_* 			// Drop intermediate variables

*===============================================================================
*------------------------SECOND STEP(DYNAMIC PANEL-GMM)-------------------------
*===============================================================================
local pf		k l 															// Production function (pf) variables (remaining part)
local markov 	L1_Rhat $markov_lag 											// Markov process variables excluding FE
if "$Uvar"!="" {
	replace Rhat = L1_$Uvar 													// Replace with endogenous variable used to test for weak identification
	replace D_Rhat = DL1_$Uvar
}

*-----------------MULTIPLY PRODUCTION INPUTS WITH exp(dt)-----------------------
qui foreach var of local pf {
	replace `var' 		= `var'*exp(dt)
	replace L1_`var'	= L.`var'
	replace L2_`var'	= L2.`var'
	replace D_`var' 	= D.`var'
	replace DL1_`var' 	= D.L1_`var'
}

*-------------INSTRUMENTS IN MARKOV PROCESS FOR EQUATION IN LEVELS--------------
macro drop Z*
gen yt_tilde = yt*exp(dt)
qui foreach var of varlist yt_tilde { 											// For second-step, GMM
	* Lags
	forvalues i = 1/3 {
		gen L`i'_`var' = L`i'.`var' 
	}
	global Zdif_yt $Zdif_yt L3_`var' 											// Use deeper lags to control for measurement error when constructing aggregate demand shifters qt
	
	* First-differences
	gen D_`var' = D.`var'
	forvalues i = 1/12 {
		gen DL`i'_`var' = L`i'.D_`var'
		if `i'>=2 global Zlev_yt $Zlev_yt DL`i'_`var' 							// Use deeper lags to control for measurement error when constructing aggregate demand shifters qt
	}
}

drop if missing(L1_Rhat) 														// Drop missing lag values (sample used for GMM)

*-------------------OLS STARTING VALUES FOR GMM-SECOND STEP---------------------
qui foreach i of local industries {
	foreach var of local pf {
	
		* Create industry specific inputs & declare locals for the OLS variables
		gen    `var'${indlev}_`i' =    `var'*${indlev}_`i'
		gen L1_`var'${indlev}_`i' = L1_`var'*${indlev}_`i'
		local OLS_var `OLS_var' `var'${indlev}_`i' L1_`var'${indlev}_`i'
		
		* Declare locals for starting values of the pf parameters
		local b_OLS `b_OLS'`comma' _b[`var'${indlev}_`i']/_b[yt_tilde]
		local comma ,
	}
}

* Declare locals for starting values of the markov process parameters
foreach var of varlist `markov' { 												// NOTE: no constant, subsumed in FE below
	local b_OLS `b_OLS'`comma' _b[`var']
}

* OLS regression
constraint 1 yt = 1
qui cnsreg Rhat `OLS_var' `markov' yt_tilde L1_yt_tilde yt L1_yt $FE, constraints(1)

* Store in a vector the starting values for pf and markov process
matrix b_OLS = `b_OLS', ln(-_b[yt_tilde])
//matrix list b_OLS

*------------VARIABLES & IV BASED ON THE TIMING ASSUMPTIONS OF INPUTS-----------
macro drop X* DX* 																// Ensure macros are empty

qui foreach i of local industries {
	foreach var of local pf {
		* X
		global X $X `var'${indlev}_`i'
		
		* X_lag
		global X_lag $X_lag L1_`var'${indlev}_`i'
		
		* DX
		gen D_`var'${indlev}_`i' = D_`var'*${indlev}_`i'
		global DX $DX D_`var'${indlev}_`i'
		
		* DX_lag
		gen DL1_`var'${indlev}_`i' = DL1_`var'*${indlev}_`i'
		global DX_lag $DX_lag DL1_`var'${indlev}_`i'
		
		* IV with PkPl (default) - Pre-determined k & Pre-determined l
		if "$inputs"=="PkPl" {
			* Zdif
			global Zdif $Zdif L1_`var'${indlev}_`i'
			
			* Zlev
			global Zlev $Zlev  D_`var'${indlev}_`i'
		}
		
		* IV with PkDl - Pre-determined k & Dynamic l 
		else if "$inputs"=="PkDl" { 
		
			if "`var'"=="k" {
				* Zdif
				global Zdif $Zdif L1_`var'${indlev}_`i'
				
				* Zlev
				global Zlev $Zlev  D_`var'${indlev}_`i'
			}
			else if "`var'"=="l" { 												// Can also add deeper lags on labour to increase IV strength
				* Zdif
				cap gen L2_`var'${indlev}_`i' = L2_`var'*${indlev}_`i'
				global Zdif $Zdif L2_`var'${indlev}_`i' 
				
				* Zlev
				cap qui gen DL1_`var'${indlev}_`i' = DL1_`var'*${indlev}_`i'
				global Zlev $Zlev DL1_`var'${indlev}_`i' //D_`var'${indlev}_`i'
			}
		}
	}
}

*-----------------------------GMM CRITERION FUNCTION----------------------------
clear mata

* Declare set of external globals to be used below
local external1 real matrix 	SX, SX_lag, SMarkov_lag, SFE, invFEpFE, Zdif, Zlev
local external2 real colvector 	SRhat, SRhat_lag, SYt, SYt_lag, SYt_tilde, SYt_tilde_lag 
local external3 real scalar 	Kx, Km, n, Ndif								

mata:
void GMM_criterion(real scalar todo, real rowvector betas, real colvector crit0, real colvector crit, real matrix S)
{
	//-------------------------------DECLARATIONS-------------------------------
	real colvector 	alpha, rho, rhoFE, FWL, SXi
	real scalar 	delta
	external 		`external1' 												// Requires less house-keeping compared to passing them as arguments to the sub-routines
	external 		`external2'
	external 		`external3'
	//--------------------------------PARAMETERS--------------------------------
	// betas is a row vector of optimisation parameters
	alpha 	= betas[|1\Kx|]' 													// Column vector for the production input parameters
	rho 	= betas[|(Kx+1)\(Kx+1+Km)|]' 										// Column vector for the rest of markov process parameters (excluding FE)
	delta 	= betas[|(Kx+Km+2)|]' 												// Scalar containing the additonal parameter to be estimated: delta
	//----------------------------MARKOV PROCESS--------------------------------
	FWL 	= SRhat + SX*(alpha*exp(delta)) + SYt_tilde*exp(delta) - SYt - ((SRhat_lag + SX_lag*(alpha*exp(delta)) + SYt_tilde_lag*exp(delta) - SYt_lag),SMarkov_lag)*rho 	// Stacked First Differences and Levels Equations: "Endogenous" part (excluding FE); for partitioning below
	rhoFE 	= invFEpFE*cross(SFE,FWL) 											// OLS to partition the exogenous FE = X1 (Frisch-Waugh-Lovell): rhoX1 = inv(X1'X1)X1'(y-X2*rhoX2)
	//--------------------------CRITERION FUNCTION------------------------------
	SXi 	= FWL - SFE*rhoFE 													// "Estimated" stacked productivity innovations (given parameter guess)
	crit = (cross(Zdif,SXi[|1\Ndif|])\cross(Zlev,SXi[|(Ndif+1)\.|]))/n 			// Sample analogue of stacked moment conditions for equations in first differences and levels
	//-----------------RE-CENTER BOOTSTRAP MOMENT FUNCTIONS---------------------
	// Horowitz and Hall (1996); Only for over-identified models (L>K)!
	if ($rep==0) crit0 = crit 													// Store moment functions with original data. Used to re-center moment functions with bootstrap data
	else 		 crit  = crit - crit0 											// Re-center bootstrap moment functions using moment functions from original data (crit0)
}
end

*-----------------------------OPTIMISATION PROCEDURE----------------------------
mata:
void GMM()
{
	//-------------------------------DECLARATIONS-------------------------------
	real scalar 	i, i1, i2, Ti, M, J 
	real rowvector 	p
	real colvector 	OTi, crit0
	real matrix 	DpZ, Zdifi, ITi, Di, W
	external 		`external1' 												// Requires less house-keeping compared to passing them as arguments to the sub-routines
	external 		`external2'
	external 		`external3'
	//--------------------------LOAD DATA STRUCTURE-----------------------------
	touse 		= (st_data(., "DL1_Rhat") :< .) 								// 0|1 identifier, where: 0 refers to missing values (and 1 otherwise) for equation in first-differences
	info 		= panelsetup(st_data(., "newid")[selectindex(touse), .], 1) 	// Set Panel structure (for the equation in first-differences) with "newid" as the panel identifier variable
	n 			= panelstats(info)[1] 											// # of firms should be the same both for equation in levels and first-differences since we kept firms with at least 3 consecutive years of info!
	Ndif 		= panelstats(info)[2] 											// # of observations for equation in first-differences
	//------------------------------LOAD DATA-----------------------------------
	SRhat 		= st_data(., "D_Rhat")[selectindex(touse), .]\st_data(.,"Rhat") // Note: for the equation in first-differences there are only T-1 observations per firm!
	SRhat_lag 	= st_data(., "DL1_Rhat")[selectindex(touse), .]\st_data(., "L1_Rhat")
	SX 			= st_data(., "$DX")[selectindex(touse), .]\st_data(.,"$X")
	SX_lag 		= st_data(., "$DX_lag")[selectindex(touse), .]\st_data(., "$X_lag")
	SMarkov_lag = st_data(., "$Dmarkov_lag")[selectindex(touse), .]\st_data(., "$markov_lag")
	SYt			= st_data(., "D_yt")[selectindex(touse), .]\st_data(.,"yt")
	SYt_lag 	= st_data(., "DL1_yt")[selectindex(touse), .]\st_data(., "L1_yt")
	SYt_tilde	= st_data(., "D_yt_tilde")[selectindex(touse), .]\st_data(.,"yt_tilde")
	SYt_tilde_lag = st_data(., "DL1_yt_tilde")[selectindex(touse), .]\st_data(., "L1_yt_tilde")
	Kx 			= cols(SX) 														// # of production input variables
	Km 			= cols(SMarkov_lag) 											// # of markov process variables (excluding persistence term and FE)
	//----------------------------FIXED EFFECTS---------------------------------
	FE 			= st_data(., "$FElist") 										// Collinear FE are identified from above and set to zero
	SFE 		= (FE-st_data(., "L.($FElist)"))[selectindex(touse), .]\FE 		// Stacked FE: drop rows with missings, i.e. no data, from equation in first differences.
	//----------------------FOR PARTITIONING THE FE-----------------------------
	invFEpFE 	= invsym(cross(SFE,SFE)) 										// Invertion of cross-product here saves time vs. inside the iterative algorithm GMM_criterion() above
	//----------------------------LOAD INSTRUMENTS------------------------------
	Zdif 		= editmissing(st_data(., "$Zdif  L2_Rhat $IVdif $Zdif_yt")[selectindex(touse), .], 0) 	// For equation in first-differences; drop rows with missings and set rest of missings to zero
	Zlev 		= editmissing(st_data(., "$Zlev DL1_Rhat $IVlev $Zlev_yt $FElist"), 0) 					// For equation in levels; set missings to zero. Over-identified model (L>K): last term adds FEs
	/*Omitting the constant/FE from the IV set makes the value and statistical
	  properties of the estimator depend on the global mean of the data, & thus
	  thus it may exhibit considerable bias and efficiency loss (especially if
	  the mean of the variable is large in magnitude). A solution is to include
	  the FE in the IV set - case for over-identified models (L>K);
	  Breusch et al. (1999); Han and Kim (2014) */
	//----------------------------WEIGHTING MATRIX------------------------------
	//--------------------------Blundell et al. 2001----------------------------
	i2 	= Ndif + n 																// Last row # of last panel i
 	DpZ = J(i2, cols(Zdif), .) 													// Empty matrix to be filled with Di'Zi
	i 	= n + 1
	while (--i) {
		Zdifi = panelsubmatrix(Zdif, i, info) 									// [(T-1)xL] sub-matrix	for panel i
		Ti 	= rows(Zdifi) 														// == T-1 # of panel i ==info[i,2]-info[i,1]+1
		OTi = J(Ti, 1, 0) 														// [Tx1] column vector
		ITi = I(Ti) 															// T-dimensional identity matrix
		Di 	= (OTi,ITi)-(ITi,OTi) 												// [(T-1)xT] first-difference transformation matrix with -1 in main diagonal and 1 in first above off-diagonal
		i1 	= i2 - Ti 															// First row # of panel i
 		DpZ[|i1,.\i2,.|] = cross(Di,Zdifi) 										// Compute Di'Zi and store in DpZ
		i2 	= i1 - 1 															// Last row # of next panel i-1
	}
	W = blockdiag(invsym(cross(DpZ,DpZ)/n),invsym(cross(Zlev,Zlev)/n))
	
	//-----------Arellano&Bover (1995); and Blundell and Bond (1998)------------
	//W=blockdiag(invsym(cross(Zdif,Zdif)/n),invsym(cross(Zlev,Zlev)/n)) 		// Classical non-robust covariance matrix
	
	//------------FOR RE-CENTERING OF THE BOOTSTRAP MOMENT FUNCTIONS------------
	if ($rep!=0) crit0 = st_matrix("crit0") 									// Load estimated sample moment functions with original sample from a Stata to MATA vector

	//------------------------SET-UP NUMERICAL METHOD---------------------------
	M = optimize_init()
	optimize_init_evaluator(M, &GMM_criterion())
	optimize_init_which(M, "min")	
	optimize_init_evaluatortype(M, "q0") 										// q evaluator for the special case of quadratic optimization: use only with techniques gn & nm
	if ($rep==0) optimize_init_params(M, st_matrix("b_OLS")) 					// Starting values from OLS
	else 		 optimize_init_params(M, st_matrix("b_boot")) 					// Starting values from original sample estimates for bootstrap sample estimations (faster compared to OLS)
	optimize_init_technique(M, "gn") 											// Optimisation technique: "gn" Gauss–Newton (quadratic optimization)
	optimize_init_gnweightmatrix(M, W) 											// Weighting matrix. If not set, default is identity matrix
	optimize_init_conv_maxiter(M, 10) 											// Max of 10 iterations: for rare non-convergence cases in weird bootstrap sample draws. Usually, converges in 3-4 iterations
	optimize_init_argument(M, 1, crit0) 										// Pass extra arguments to evaluator: crit0 is vector with moment functions from original data
	p = optimize(M)
	//------------------HANSEN-J OVERIDENTIFICATION STATISTIC-------------------
	J = n*optimize_result_value(M) 												// Using fixed weighting matrix for the one-step GMM estimation
 	//----------------------------EXPORT TO STATA-------------------------------
	st_matrix("J", J)
	st_matrix("b_gmm", p)
	if ($rep==0) {
		st_matrix("crit0", crit0) 												// Store moment functions with original sample to a Stata vector.
		st_matrix("b_boot", p) 													// Original sample estimates as starting values for bootstrap estimations (runs faster vs. OLS starting values)
	}
}
end

*---------------------------------RUN ALGORITHM---------------------------------
mata GMM()
//matrix list b_gmm 															// Display GMM estimates
//matrix list J 																// Display Hansen-J statistic (for over-identified models;L>K)
clear mata 																		// Save memory

*===============================================================================
*-----------------------------------RESULTS-------------------------------------
*===============================================================================

*----------------------------REMAINING PF COEFFICIENTS--------------------------
local n = 1
local plus 																		// Ensure local is empty
qui foreach i of local industries {
	foreach var of local pf {
		local b_`var' `b_`var''`plus'b_gmm[1,`n']*${indlev}_`i'
		local ++n
	}
	local plus +
}

* Remaining pf input parameters
qui foreach var of local pf {
	gen b_`var' = `b_`var''
}

*--------------------------MARKOV PROCESS COEFFICIENTS--------------------------
qui foreach var in omega $markov {
	gen rho_`var' = b_gmm[1,`n']
	local ++n 																	// Keep counting from above
}

*-----------------------------------MARKUPS-------------------------------------
qui gen delta 	= b_gmm[1,`n']
qui gen markup 	= 1/exp(dt+delta) 												// Aggregate time-varying markups
qui gen sigma 	= 1/(exp(dt+delta)-1) 											// Time-varying elasticity of demand


*---------------------HANSEN-J OVERIDENTIFICATION STATISTIC---------------------
gen rho_J = J[1,1]

*------------------------OUTPUT ELASTICITIES, RTS & TFP-------------------------
* Output elasticities
qui gen a_k = - b_k 															// For Cobb-Douglas pf they coincide with the respective pf parameters
qui gen a_l = - b_l
qui gen a_m = a_m_delta/exp(delta)

* Returns To Scale
qui gen RTS 	 = a_k + a_l + a_m
qui gen RTS_test = RTS - 1 														// Test for Constant RTS (CRTS)

* TFP
qui foreach var in "" L1_ {
	gen `var'omega_c 	= `var'Rhat + (b_k*`var'k + b_l*`var'l)*exp(dt+delta) + (exp(dt+delta)-1)*yt
	gen `var'omega 		= `var'omega_c + `var'eit 								// Not corrected for ex-post (transitory) shocks to production
}

*----------------------------------HOUSEKEEPING---------------------------------
drop *${indlev}_* *hat* *eit* b_* yt *_yt *delta dt
matrix drop b_OLS b_gmm J
//sum a_* rho_* omega
